/**
 * AskEngine Background Service Worker
 * 
 * Routes queries through the AskEngine proxy (free tier, 20 queries)
 * or directly to Anthropic if user has their own API key.
 */

const PROXY_URL = 'https://noisy-shadow-b288.pengusaurusrex.workers.dev'; // ← Update after deploying worker
const FREE_LIMIT = 10;

const DEFAULTS = {
  enabled: true,
  apiKey: '',
  model: 'claude-sonnet-4-20250514',
  maxTokens: 1024,
  position: 'bottom-left'
};

// ── Install & UID ──
chrome.runtime.onInstalled.addListener(async () => {
  const existing = await chrome.storage.sync.get(Object.keys(DEFAULTS));
  const toSet = {};
  for (const [key, value] of Object.entries(DEFAULTS)) {
    if (existing[key] === undefined) toSet[key] = value;
  }
  if (Object.keys(toSet).length > 0) await chrome.storage.sync.set(toSet);

  const { uid } = await chrome.storage.local.get('uid');
  if (!uid) {
    const newUid = 'ae_' + crypto.randomUUID().replace(/-/g, '').slice(0, 24);
    await chrome.storage.local.set({ uid: newUid, queryCount: 0 });
  }
});

// ── Message Handling ──
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'ASKENGINE_QUERY') {
    handleQuery(message.payload).then(sendResponse).catch(err => {
      sendResponse({ error: err.message || 'Unknown error' });
    });
    return true;
  }
  if (message.type === 'GET_SETTINGS') {
    getSettingsWithUsage().then(sendResponse);
    return true;
  }
  if (message.type === 'SAVE_SETTINGS') {
    chrome.storage.sync.set(message.payload).then(() => sendResponse({ ok: true }));
    return true;
  }
  if (message.type === 'GET_USAGE') {
    getUsageInfo().then(sendResponse);
    return true;
  }
  if (message.type === 'VERIFY_PAYMENT') {
    verifyPayment().then(sendResponse);
    return true;
  }
});

async function verifyPayment() {
  const { uid } = await chrome.storage.local.get('uid');
  try {
    const resp = await fetch(`${PROXY_URL}/v1/verify-payment`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ uid })
    });
    const data = await resp.json();
    if (data.paid) {
      await chrome.storage.local.set({ paid: true, paidUntil: data.paidUntil });
    }
    return data;
  } catch(e) {
    return { paid: false, error: e.message };
  }
}

async function getSettingsWithUsage() {
  const settings = await chrome.storage.sync.get(null);
  const local = await chrome.storage.local.get(['uid', 'queryCount']);
  return {
    ...settings,
    uid: local.uid,
    queryCount: local.queryCount || 0,
    freeLimit: FREE_LIMIT,
    hasOwnKey: !!settings.apiKey,
    unlimited: !!settings.apiKey,
  };
}

async function getUsageInfo() {
  const { apiKey } = await chrome.storage.sync.get('apiKey');
  const { queryCount } = await chrome.storage.local.get('queryCount');
  return {
    used: queryCount || 0,
    limit: FREE_LIMIT,
    remaining: apiKey ? Infinity : Math.max(0, FREE_LIMIT - (queryCount || 0)),
    unlimited: !!apiKey,
  };
}

// ═══════════════════════════════════════════════════════════
// QUERY HANDLER
// ═══════════════════════════════════════════════════════════
async function handleQuery(payload) {
  const { query, pageContext, conversationHistory, siteType } = payload;
  const settings = await chrome.storage.sync.get(['apiKey', 'model', 'maxTokens']);
  const { uid, queryCount, paid, paidUntil } = await chrome.storage.local.get(['uid', 'queryCount', 'paid', 'paidUntil']);

  const hasOwnKey = !!settings.apiKey;
  const isPaid = paid && paidUntil && new Date(paidUntil) > new Date();
  const currentCount = queryCount || 0;

  if (!hasOwnKey && !isPaid && currentCount >= FREE_LIMIT) {
    return {
      error: null, answer: null, limitReached: true,
      usage: { used: currentCount, limit: FREE_LIMIT, remaining: 0 }
    };
  }

  const systemPrompt = buildSystemPrompt(pageContext, siteType);
  const messages = [];
  if (conversationHistory && conversationHistory.length > 0) {
    for (const msg of conversationHistory.slice(0, -1)) {
      messages.push({ role: msg.role === 'user' ? 'user' : 'assistant', content: msg.content });
    }
  }
  messages.push({ role: 'user', content: query });

  try {
    let answer;
    if (hasOwnKey) {
      answer = await callAnthropicDirect(settings, systemPrompt, messages);
    } else {
      answer = await callProxy(uid, systemPrompt, messages, settings.model, settings.maxTokens);
    }

    if (!hasOwnKey && !isPaid) {
      await chrome.storage.local.set({ queryCount: currentCount + 1 });
    }

    const unlimited = hasOwnKey || isPaid;
    return {
      answer, error: null,
      usage: {
        used: unlimited ? 0 : currentCount + 1,
        limit: unlimited ? Infinity : FREE_LIMIT,
        remaining: unlimited ? Infinity : Math.max(0, FREE_LIMIT - currentCount - 1),
        unlimited,
        paid: isPaid,
      }
    };
  } catch (e) {
    if (e.message?.includes('free_limit_reached')) {
      return { error: null, answer: null, limitReached: true, usage: { used: FREE_LIMIT, limit: FREE_LIMIT, remaining: 0 } };
    }
    if (e.message?.includes('Failed to fetch')) {
      return { error: 'Network error. Check your internet connection.', answer: null };
    }
    return { error: e.message || 'Unexpected error', answer: null };
  }
}

async function callAnthropicDirect(settings, system, messages) {
  const response = await fetch('https://api.anthropic.com/v1/messages', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'x-api-key': settings.apiKey,
      'anthropic-version': '2023-06-01',
      'anthropic-dangerous-direct-browser-access': 'true'
    },
    body: JSON.stringify({
      model: settings.model || DEFAULTS.model,
      max_tokens: settings.maxTokens || DEFAULTS.maxTokens,
      system, messages
    })
  });
  if (!response.ok) {
    const err = await response.json().catch(() => ({}));
    if (response.status === 401) throw new Error('Invalid API key. Check your settings.');
    if (response.status === 429) throw new Error('Rate limited. Please wait a moment.');
    throw new Error(err.error?.message || `API error (${response.status})`);
  }
  const data = await response.json();
  return data.content?.filter(c => c.type === 'text')?.map(c => c.text)?.join('\n') || 'No response.';
}

async function callProxy(uid, system, messages, model, maxTokens) {
  const response = await fetch(`${PROXY_URL}/v1/query`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ uid, system, messages, model: model || DEFAULTS.model, max_tokens: maxTokens || DEFAULTS.maxTokens })
  });
  const data = await response.json();
  if (!response.ok) {
    if (data.error === 'free_limit_reached') throw new Error('free_limit_reached');
    throw new Error(data.error || `Proxy error (${response.status})`);
  }
  if (data.usage) await chrome.storage.local.set({ queryCount: data.usage.used });
  return data.answer;
}

// ═══════════════════════════════════════════════════════════
// SYSTEM PROMPT
// ═══════════════════════════════════════════════════════════
function buildSystemPrompt(pageContext, siteType) {
  const itemsSummary = pageContext.items
    ? `\n\nITEMS ON PAGE (${pageContext.itemCount || pageContext.items.length} total):\n${JSON.stringify(pageContext.items, null, 0)}`
    : '';
  const contentSummary = pageContext.content ? `\n\nPAGE CONTENT:\n${pageContext.content}` : '';
  const structuredSummary = pageContext.structuredData ? `\n\nSTRUCTURED DATA:\n${JSON.stringify(pageContext.structuredData, null, 0)}` : '';

  return `You are AskEngine, a conversational search assistant embedded on a webpage. You help users find and understand content on the page they're currently viewing.

CURRENT PAGE CONTEXT:
- URL: ${pageContext.url || 'unknown'}
- Title: ${pageContext.title || 'unknown'}
- Site type: ${siteType || 'generic'}
${pageContext.heading ? '- Main heading: ' + pageContext.heading : ''}
${pageContext.breadcrumbs ? '- Breadcrumbs: ' + pageContext.breadcrumbs : ''}
${pageContext.totalResults ? '- Total results on page: ' + pageContext.totalResults : ''}
${pageContext.desc ? '- Description: ' + pageContext.desc : ''}
${itemsSummary}${contentSummary}${structuredSummary}

INSTRUCTIONS:
1. Answer questions about the page's content naturally and helpfully.
2. When the user asks about products/items/listings, reference the actual items found on the page by name, price, and other attributes.
3. If comparing items, be specific about which items you're comparing and why.
4. If the user asks for something not on the page, say so clearly but suggest alternatives from what IS available.
5. Keep responses concise but informative — you're embedded in a small widget.
6. When recommending items, mention the item name clearly so the user can find it on the page.
7. If asked to summarize, focus on the most useful/actionable information.
8. Never make up items or prices — only reference what's actually in the page data.
9. If the page has very little data, acknowledge the limitation and offer what help you can.
10. For non-product pages (articles, docs, wikis, social feeds), provide intelligent summarization and Q&A.

CRITICAL FORMATTING RULES:
- ALWAYS wrap item/product names in **double asterisks** (bold). This makes them clickable links to the product page in the widget.
- Use a RECOGNIZABLE portion of the exact item name from the data — at least 4-5 distinctive words. Do NOT paraphrase into generic terms.
- EVERY product/item you mention MUST be wrapped in **bold**. No exceptions.
- Put the price right after the bold name, like: **Item Name** for $3.99
- Keep paragraphs short (2-3 sentences max)
- Use line breaks between items when listing multiple
- Don't use headers or complex markdown — this displays in a chat bubble`;
}
