/**
 * AskEngine Content Script
 * Runs in Chrome's isolated world. Extractor is inlined — no script injection needed.
 */
(function() {
  'use strict';
  if (window.__askengine_loaded) return;
  window.__askengine_loaded = true;

  let pageData = null;
  let isOpen = false;
  let widgetShadow = null;
  let conversationHistory = [];

  // =======================================================================
  // EXTRACTOR (inlined — runs in content script world, full DOM access)
  // =======================================================================
  const Extractor = {
    siteType: 'generic',

    async extractPageContent() {
      const url = window.location.href;
      this.siteType = this.detectSiteType(window.location.hostname);
      const result = {
        url, hostname: window.location.hostname, siteType: this.siteType,
        title: document.title, items: [], content: null,
        meta: this.extractMeta(), navigation: this.extractNav()
      };
      const structured = this.extractJsonLd();
      if (structured.length > 0) result.structured = structured;
      const adapted = this.runAdapter();
      if (adapted.length > 0) result.items = adapted;
      if (result.items.length < 3) result.items = this.dedupe(result.items, this.extractCards());
      if (result.items.length < 3) result.items = this.dedupe(result.items, this.extractRepeated());
      if (result.items.length < 3) result.items = this.dedupe(result.items, this.extractTables());
      
      // Layer 6: Universal product-link scanner (works on any site with product URLs)
      if (result.items.length < 3) result.items = this.dedupe(result.items, this.extractProductLinks());
      
      result.content = this.extractMainText();
      return result;
    },

    detectSiteType(h) {
      h = h.toLowerCase();
      if (h.includes('amazon.')) return 'amazon';
      if (h.includes('ebay.')) return 'ebay';
      if (h.includes('etsy.')) return 'etsy';
      if (h.includes('walmart.')) return 'walmart';
      if (h.includes('temu.')) return 'temu';
      if (h.includes('aliexpress.')) return 'aliexpress';
      if (h.includes('target.')) return 'target';
      if (document.querySelector('meta[name="shopify-checkout-api-token"],link[href*="cdn.shopify.com"]')) return 'shopify';
      if (document.querySelector('.woocommerce-page,link[href*="woocommerce"]')) return 'woocommerce';
      if (h.includes('zillow.')) return 'zillow';
      if (h.includes('indeed.')) return 'indeed';
      if (h.includes('linkedin.')) return 'linkedin';
      if (h.includes('yelp.')) return 'yelp';
      if (h.includes('reddit.')) return 'reddit';
      if (h.includes('wikipedia.')) return 'wikipedia';
      if (h.includes('github.')) return 'github';
      if (h.includes('airbnb.')) return 'airbnb';
      if (h.includes('booking.com')) return 'booking';
      // Social & content platforms
      if (h.includes('x.com') || h.includes('twitter.com')) return 'twitter';
      if (h.includes('youtube.com') || h.includes('youtu.be')) return 'youtube';
      if (h.includes('news.ycombinator.com')) return 'hackernews';
      if (h.includes('stackoverflow.com') || h.includes('stackexchange.com')) return 'stackoverflow';
      if (h.includes('medium.com') || document.querySelector('meta[property="al:android:package"][content="com.medium.reader"]')) return 'medium';
      // News site detection
      if (document.querySelector('meta[property="og:type"][content="article"],meta[name="article:published_time"],meta[property="article:published_time"]')) return 'article';
      return 'generic';
    },

    extractJsonLd() {
      const r = [];
      document.querySelectorAll('script[type="application/ld+json"]').forEach(s => {
        try {
          const d = JSON.parse(s.textContent);
          const items = Array.isArray(d) ? d : d['@graph'] ? d['@graph'] : [d];
          items.forEach(item => { if (item['@type']) r.push({type:item['@type'], data:this.flat(item)}); });
        } catch(e) {}
      });
      return r;
    },

    flat(obj, pfx='', dep=0) {
      if (dep > 4) return {};
      const f = {};
      for (const [k,v] of Object.entries(obj)) {
        if (k.startsWith('@') && k !== '@type') continue;
        const key = pfx ? pfx+'.'+k : k;
        if (v && typeof v === 'object' && !Array.isArray(v)) Object.assign(f, this.flat(v, key, dep+1));
        else if (Array.isArray(v)) f[key] = v.slice(0,5).map(x => typeof x==='object' ? JSON.stringify(x).slice(0,150) : String(x)).join(', ');
        else if (v != null) f[key] = String(v).slice(0,300);
      }
      return f;
    },

    extractMeta() {
      const m = {};
      const d = document.querySelector('meta[name="description"]');
      if (d) m.description = d.getAttribute('content');
      document.querySelectorAll('meta[property^="og:"]').forEach(el => {
        m['og_'+el.getAttribute('property').replace('og:','')] = el.getAttribute('content');
      });
      return Object.keys(m).length > 0 ? m : null;
    },

    extractNav() {
      const n = {};
      const h1 = document.querySelector('h1');
      if (h1) n.heading = h1.textContent.trim().slice(0,200);
      const bc = document.querySelector('[class*="breadcrumb"],[aria-label="breadcrumb"]');
      if (bc) n.breadcrumbs = Array.from(bc.querySelectorAll('a,span,li')).map(e=>e.textContent.trim()).filter(Boolean).slice(0,8);
      const cm = document.body.textContent.match(/(\d[\d,]*)\s*(?:results?|items?|products?)/i);
      if (cm) n.totalResults = cm[0];
      return Object.keys(n).length > 0 ? n : null;
    },

    runAdapter() {
      try {
        const a = {
          amazon: () => this.aAmazon(), temu: () => this.aTemu(), ebay: () => this.aEbay(),
          shopify: () => this.aShopify(), woocommerce: () => this.aWoo(),
          indeed: () => this.aIndeed(), reddit: () => this.aReddit(),
          twitter: () => this.aTwitter(), youtube: () => this.aYouTube(),
          wikipedia: () => this.aWikipedia(), hackernews: () => this.aHackerNews(),
          stackoverflow: () => this.aStackOverflow(), article: () => this.aArticle(),
          medium: () => this.aArticle()
        };
        return (a[this.siteType] || (() => []))();
      } catch(e) { return []; }
    },

    aAmazon() {
      const items = [];
      document.querySelectorAll('[data-component-type="s-search-result"],.s-result-item').forEach(c => {
        const name = c.querySelector('h2 a span,.a-text-normal')?.textContent?.trim();
        if (!name) return;
        items.push(this.norm({name, price:c.querySelector('.a-price .a-offscreen')?.textContent?.trim(),
          rating:c.querySelector('.a-icon-alt')?.textContent?.trim(), image:c.querySelector('.s-image')?.src,
          link:c.querySelector('h2 a')?.href}));
      });
      if (!items.length) {
        const n = document.querySelector('#productTitle')?.textContent?.trim();
        if (n) items.push(this.norm({name:n, price:document.querySelector('.a-price .a-offscreen')?.textContent?.trim(),
          rating:document.querySelector('#acrPopover')?.getAttribute('title'),
          description:document.querySelector('#feature-bullets')?.textContent?.trim()?.slice(0,400)}));
      }
      return items.filter(Boolean);
    },

    aTemu() {
      const items = [];
      const seen = new Set();

      // Strategy 1: Find ALL product links — Temu product URLs contain numeric IDs
      // This is the most reliable approach since class names are obfuscated
      document.querySelectorAll('a[href]').forEach(a => {
        const href = a.href || '';
        // Temu product URLs: /goods-detail, /product, or URLs with numeric goods IDs
        if (!href.match(/\/(goods|product|[a-z]+-[a-z]+-).*\.html|\/\d{5,}/)) return;
        if (href.includes('/cart') || href.includes('/login') || href.includes('/coupon')) return;

        // Walk up to find the product card container (usually 2-4 levels up)
        let container = a;
        for (let i = 0; i < 5; i++) {
          if (!container.parentElement) break;
          const parent = container.parentElement;
          // Stop if we hit a big grid/list container
          const kids = parent.children.length;
          if (kids > 15) break;
          container = parent;
        }

        // Extract name — try multiple strategies
        let name = null;
        // 1. Image alt text (Temu almost always has descriptive alt text)
        const imgs = container.querySelectorAll('img[alt]');
        for (const img of imgs) {
          const alt = img.alt?.trim();
          if (alt && alt.length > 8 && alt.length < 300 && !alt.match(/^(logo|icon|banner|ad)/i)) {
            name = alt;
            break;
          }
        }
        // 2. Title attribute on the link itself
        if (!name) name = a.title?.trim() || a.getAttribute('aria-label')?.trim();
        // 3. Any heading inside
        if (!name) name = container.querySelector('h1,h2,h3,h4,h5,h6')?.textContent?.trim();
        // 4. Longest text node that looks like a product title
        if (!name) {
          let bestText = '';
          container.querySelectorAll('span,div,p').forEach(el => {
            const t = el.textContent?.trim();
            if (t && t.length > 10 && t.length < 200 && t.length > bestText.length &&
                !t.match(/^[\d\s\.,₹$€£¥%]+$/) && !t.match(/^(save|sold|free|ship|off)/i) &&
                el.children.length === 0) {
              bestText = t;
            }
          });
          if (bestText.length > 10) name = bestText;
        }

        if (!name || name.length < 5) return;
        const nameKey = name.toLowerCase().slice(0, 40);
        if (seen.has(nameKey)) return;
        seen.add(nameKey);

        // Extract price — find text that looks like a price within/near the container
        let price = null;
        const priceTexts = container.textContent.match(/[\d,]+[LlL]\b|\$\s?\d[\d,.]*|€\s?\d[\d,.]*|£\s?\d[\d,.]*/g);
        if (priceTexts && priceTexts.length > 0) {
          // Take the first (usually the main price, not the RRP)
          price = priceTexts[0];
        }

        // Extract sold count
        let sold = null;
        const soldMatch = container.textContent.match(/[\d,.]+[Kk]?\+?\s*sold/i);
        if (soldMatch) sold = soldMatch[0];

        // Extract rating
        let rating = null;
        const ratingEl = container.querySelector('[aria-label*="star"],[aria-label*="rating"]');
        if (ratingEl) rating = ratingEl.getAttribute('aria-label');
        else {
          const starMatch = container.textContent.match(/[★☆⭐]\s*[\d.]+|[\d.]+\s*(?:\/\s*5|stars?)/i);
          if (starMatch) rating = starMatch[0];
        }

        // Image
        let image = null;
        const imgEl = container.querySelector('img[src*="img.kwcdn"],img[src*="img.ltwebstatic"],img[src]');
        if (imgEl) image = imgEl.src;

        items.push(this.norm({ name, price, link: href, image, sold, rating }));
      });

      // Strategy 2: If strategy 1 found very little, scan ALL images with good alt text
      if (items.length < 5) {
        document.querySelectorAll('img[alt]').forEach(img => {
          const alt = img.alt?.trim();
          if (!alt || alt.length < 10 || alt.length > 300) return;
          if (alt.match(/^(logo|icon|banner|ad|arrow|close|menu)/i)) return;
          const nameKey = alt.toLowerCase().slice(0, 40);
          if (seen.has(nameKey)) return;
          seen.add(nameKey);

          // Find nearest link
          const link = img.closest('a[href]')?.href;
          if (!link) return;

          // Find nearest price text
          const parent = img.closest('div,li,section') || img.parentElement;
          let price = null;
          if (parent) {
            const pm = parent.textContent.match(/[\d,]+[LlL]\b|\$\s?\d[\d,.]*|€\s?\d[\d,.]*|£\s?\d[\d,.]*/);
            if (pm) price = pm[0];
          }

          items.push(this.norm({ name: alt, price, link, image: img.src }));
        });
      }

      return items.filter(Boolean).slice(0, 50);
    },

    aEbay() {
      const items = [];
      document.querySelectorAll('.s-item').forEach(c => {
        const name = c.querySelector('.s-item__title')?.textContent?.trim();
        if (!name || name === 'Shop on eBay') return;
        items.push(this.norm({name, price:c.querySelector('.s-item__price')?.textContent?.trim(),
          image:c.querySelector('.s-item__image img')?.src, link:c.querySelector('.s-item__link')?.href}));
      });
      return items.filter(Boolean);
    },

    aShopify() {
      const items = [];
      const seen = new Set();

      // Strategy 1: Product JSON (single product pages)
      const pj = document.querySelector('script[data-product-json],script#ProductJson');
      if (pj) { try { const p = JSON.parse(pj.textContent); items.push(this.norm({name:p.title,
        price:p.price?'$'+(p.price/100).toFixed(2):null, description:p.description?.replace(/<[^>]*>/g,'')?.slice(0,300),
        image:p.featured_image, vendor:p.vendor})); seen.add(p.title?.toLowerCase()?.slice(0,40));
      } catch(e) {} }

      // Strategy 2: Standard Shopify card classes
      document.querySelectorAll('.product-card,.grid-product,[class*="ProductCard"],[class*="product-card"],[class*="product-grid-item"],[class*="collection-product"]').forEach(c => {
        const item = this.cardGeneric(c);
        if (item && !seen.has(item.name.toLowerCase().slice(0,40))) {
          seen.add(item.name.toLowerCase().slice(0,40));
          items.push(item);
        }
      });

      // Strategy 3: Find all links to /products/ pages (works on ANY Shopify theme)
      if (items.length < 3) {
        document.querySelectorAll('a[href*="/products/"]').forEach(a => {
          const href = a.href;
          if (!href || href.includes('/products/') === false) return;
          if (href.includes('#') || href.includes('?variant') || href.endsWith('/products/') || href.endsWith('/products')) return;

          // Walk up to find the product card container
          let container = a;
          for (let i = 0; i < 5; i++) {
            if (!container.parentElement) break;
            const parent = container.parentElement;
            if (parent.children.length > 20) break;
            container = parent;
          }

          // Name: heading > link text > image alt
          let name = container.querySelector('h1,h2,h3,h4,h5,h6')?.textContent?.trim();
          if (!name || name.length < 3) {
            const imgs = container.querySelectorAll('img[alt]');
            for (const img of imgs) {
              const alt = img.alt?.trim();
              if (alt && alt.length > 3 && alt.length < 200 && !alt.match(/^(logo|icon|arrow|close)/i)) { name = alt; break; }
            }
          }
          if (!name || name.length < 3) name = a.textContent?.trim()?.slice(0, 150);
          if (!name || name.length < 3) return;

          const nameKey = name.toLowerCase().slice(0,40);
          if (seen.has(nameKey)) return;
          seen.add(nameKey);

          // Price
          let price = null;
          const priceEl = container.querySelector('[class*="price"],[class*="Price"],[data-price]');
          if (priceEl) price = priceEl.textContent.trim();
          else {
            const pm = container.textContent.match(/\$\s?\d[\d,.]*|€\s?\d[\d,.]*|£\s?\d[\d,.]*/);
            if (pm) price = pm[0];
          }

          // Image
          let image = null;
          const imgEl = container.querySelector('img[src]');
          if (imgEl) image = imgEl.src;

          // Color/variant info
          let variant = null;
          const colorEl = container.querySelector('[class*="color"],[class*="variant"],[class*="swatch"]');
          if (colorEl) variant = colorEl.textContent?.trim()?.slice(0, 60);

          items.push(this.norm({ name, price, link: href, image, variant }));
        });
      }

      return items.filter(Boolean);
    },

    aWoo() {
      const items = [];
      document.querySelectorAll('li.product,.type-product').forEach(c => {
        const name = c.querySelector('.woocommerce-loop-product__title,h2,h3')?.textContent?.trim();
        if (!name) return;
        items.push(this.norm({name, price:c.querySelector('.price,.woocommerce-Price-amount')?.textContent?.trim(),
          image:c.querySelector('img')?.src, link:c.querySelector('a[href]')?.href}));
      });
      return items.filter(Boolean);
    },

    aIndeed() {
      const items = [];
      document.querySelectorAll('.job_seen_beacon,[class*="jobCard"],.resultContent').forEach(c => {
        const t = c.querySelector('.jobTitle a,[class*="jobTitle"] a,h2 a')?.textContent?.trim();
        if (!t) return;
        items.push(this.norm({name:t, company:c.querySelector('.companyName,[data-testid="company-name"]')?.textContent?.trim(),
          location:c.querySelector('.companyLocation,[data-testid="text-location"]')?.textContent?.trim(),
          salary:c.querySelector('[class*="salary"]')?.textContent?.trim(),
          description:c.querySelector('.job-snippet,[class*="snippet"]')?.textContent?.trim()?.slice(0,200)}));
      });
      return items.filter(Boolean);
    },

    aReddit() {
      const items = [];
      document.querySelectorAll('shreddit-post,.Post,.thing.link,[data-testid="post-container"],article').forEach(p => {
        const t = p.querySelector('a[slot="title"],[data-testid="post-title"],a.title,h3')?.textContent?.trim() || p.getAttribute('post-title');
        if (!t) return;
        items.push(this.norm({name:t,
          author:p.querySelector('[data-testid="post-author"],.author')?.textContent?.trim()||p.getAttribute('author'),
          score:p.getAttribute('score')||p.querySelector('[class*="score"]')?.textContent?.trim()}));
      });
      return items.filter(Boolean);
    },

    // ── Twitter/X ──
    aTwitter() {
      const items = [];
      const seen = new Set();
      // X uses article elements for each tweet in the timeline
      document.querySelectorAll('article[data-testid="tweet"],article[role="article"]').forEach(tweet => {
        // Author
        const authorEl = tweet.querySelector('[data-testid="User-Name"],a[role="link"][href*="/"]');
        let author = null;
        if (authorEl) {
          const spans = authorEl.querySelectorAll('span');
          for (const s of spans) {
            const t = s.textContent?.trim();
            if (t && t.length > 1 && !t.startsWith('·') && !t.includes('…')) { author = t; break; }
          }
        }

        // Tweet text
        const textEl = tweet.querySelector('[data-testid="tweetText"]');
        const text = textEl?.textContent?.trim()?.slice(0, 280) || '';
        if (!text || text.length < 3) return;

        const key = text.slice(0, 50).toLowerCase();
        if (seen.has(key)) return;
        seen.add(key);

        // Engagement metrics
        const metrics = {};
        tweet.querySelectorAll('[data-testid="reply"],[data-testid="retweet"],[data-testid="like"],[data-testid="bookmark"]').forEach(btn => {
          const label = btn.getAttribute('aria-label') || '';
          if (label) {
            const testId = btn.getAttribute('data-testid');
            if (testId === 'reply') metrics.replies = label;
            if (testId === 'retweet') metrics.retweets = label;
            if (testId === 'like') metrics.likes = label;
          }
        });

        // Timestamp
        const time = tweet.querySelector('time')?.getAttribute('datetime') || tweet.querySelector('time')?.textContent?.trim();

        // Link to tweet
        const link = tweet.querySelector('a[href*="/status/"]')?.href;

        // Media
        const hasMedia = !!tweet.querySelector('img[src*="pbs.twimg"],video,[data-testid="tweetPhoto"]');

        items.push(this.norm({
          name: text.slice(0, 120) + (text.length > 120 ? '...' : ''),
          author, time, link,
          description: text.length > 120 ? text : undefined,
          likes: metrics.likes, retweets: metrics.retweets, replies: metrics.replies,
          hasMedia: hasMedia ? 'yes' : undefined
        }));
      });
      return items.filter(Boolean);
    },

    // ── YouTube ──
    aYouTube() {
      const items = [];
      const path = window.location.pathname;

      if (path.startsWith('/watch')) {
        // Single video page
        const title = document.querySelector('h1.ytd-watch-metadata yt-formatted-string, h1.ytd-video-primary-info-renderer')?.textContent?.trim();
        const channel = document.querySelector('#channel-name a, ytd-channel-name a')?.textContent?.trim();
        const views = document.querySelector('#info-strings yt-formatted-string, .view-count, [class*="view-count"]')?.textContent?.trim();
        const description = document.querySelector('#description-inner, #snippet-text, ytd-text-inline-expander')?.textContent?.trim()?.slice(0, 500);
        const likes = document.querySelector('[aria-label*="like"], #segmented-like-button button[aria-label]')?.getAttribute('aria-label');

        if (title) {
          items.push(this.norm({ name: title, author: channel, views, description, likes }));
        }

        // Related/recommended videos
        document.querySelectorAll('ytd-compact-video-renderer, ytd-rich-item-renderer').forEach(v => {
          const vTitle = v.querySelector('#video-title')?.textContent?.trim();
          if (!vTitle) return;
          const vChannel = v.querySelector('.ytd-channel-name a, #channel-name')?.textContent?.trim();
          const vViews = v.querySelector('#metadata-line span')?.textContent?.trim();
          const vLink = v.querySelector('a[href*="/watch"]')?.href;
          items.push(this.norm({ name: vTitle, author: vChannel, views: vViews, link: vLink }));
        });
      } else {
        // Browse/search/home page
        document.querySelectorAll('ytd-video-renderer, ytd-rich-item-renderer, ytd-grid-video-renderer').forEach(v => {
          const vTitle = v.querySelector('#video-title')?.textContent?.trim();
          if (!vTitle) return;
          const vChannel = v.querySelector('.ytd-channel-name a, #channel-name, #text.ytd-channel-name')?.textContent?.trim();
          const vViews = v.querySelector('#metadata-line span')?.textContent?.trim();
          const vLink = v.querySelector('a[href*="/watch"]')?.href;
          items.push(this.norm({ name: vTitle, author: vChannel, views: vViews, link: vLink }));
        });
      }
      return items.filter(Boolean);
    },

    // ── Wikipedia ──
    aWikipedia() {
      const items = [];
      // Extract article sections as items for navigation
      const sections = document.querySelectorAll('#mw-content-text h2 .mw-headline, #mw-content-text h2');
      sections.forEach(h => {
        const name = h.textContent?.trim()?.replace(/\[edit\]/g, '').trim();
        if (!name || name === 'Contents' || name === 'References' || name === 'External links' || name === 'See also' || name === 'Notes' || name === 'Further reading') return;
        // Get the first paragraph after this section
        let desc = '';
        let el = h.closest('h2')?.nextElementSibling;
        while (el && !['H2','H3'].includes(el.tagName)) {
          if (el.tagName === 'P' && el.textContent.trim().length > 20) {
            desc = el.textContent.trim().slice(0, 200);
            break;
          }
          el = el.nextElementSibling;
        }
        items.push(this.norm({ name, description: desc || undefined }));
      });

      // Also extract the lead paragraph
      const lead = document.querySelector('#mw-content-text .mw-parser-output > p:not(.mw-empty-elt)');
      if (lead && lead.textContent.trim().length > 50) {
        items.unshift(this.norm({ name: 'Overview', description: lead.textContent.trim().slice(0, 400) }));
      }
      return items.filter(Boolean);
    },

    // ── Hacker News ──
    aHackerNews() {
      const items = [];
      document.querySelectorAll('.athing').forEach(row => {
        const titleEl = row.querySelector('.titleline a');
        if (!titleEl) return;
        const name = titleEl.textContent?.trim();
        const link = titleEl.href;
        const subtext = row.nextElementSibling;
        const score = subtext?.querySelector('.score')?.textContent?.trim();
        const author = subtext?.querySelector('.hnuser')?.textContent?.trim();
        const comments = subtext?.querySelector('a[href*="item"]')?.textContent?.trim();
        const age = subtext?.querySelector('.age')?.textContent?.trim();
        items.push(this.norm({ name, link, score, author, comments, time: age }));
      });
      return items.filter(Boolean);
    },

    // ── StackOverflow ──
    aStackOverflow() {
      const items = [];
      // Question list pages
      document.querySelectorAll('.s-post-summary, .question-summary').forEach(q => {
        const title = q.querySelector('.s-post-summary--content-title a, .s-link, h3 a')?.textContent?.trim();
        if (!title) return;
        const votes = q.querySelector('.s-post-summary--stats-item-number, .vote-count-post')?.textContent?.trim();
        const answers = q.querySelectorAll('.s-post-summary--stats-item-number, .status')[1]?.textContent?.trim();
        const tags = Array.from(q.querySelectorAll('.post-tag, .s-tag')).map(t => t.textContent.trim()).join(', ');
        const link = q.querySelector('.s-post-summary--content-title a, h3 a')?.href;
        items.push(this.norm({ name: title, votes, answers, tags, link }));
      });
      // Single question page
      if (items.length === 0) {
        const qTitle = document.querySelector('#question-header h1 a, .question-hyperlink')?.textContent?.trim();
        if (qTitle) {
          const qBody = document.querySelector('.s-prose, .post-text')?.textContent?.trim()?.slice(0, 300);
          const votes = document.querySelector('.js-vote-count, [itemprop="upvoteCount"]')?.textContent?.trim();
          items.push(this.norm({ name: qTitle, description: qBody, votes }));
        }
        // Answers
        document.querySelectorAll('.answer, #answers .s-prose').forEach((a, i) => {
          const body = a.querySelector('.s-prose, .post-text')?.textContent?.trim()?.slice(0, 200);
          const votes = a.querySelector('.js-vote-count, [itemprop="upvoteCount"]')?.textContent?.trim();
          const accepted = !!a.querySelector('.accepted-answer, .js-accepted-answer-indicator');
          if (body) items.push(this.norm({ name: `Answer ${i+1}${accepted ? ' ✓ Accepted' : ''}`, description: body, votes }));
        });
      }
      return items.filter(Boolean);
    },

    // ── Article / News Pages ──
    aArticle() {
      const items = [];
      // Extract article metadata
      const title = document.querySelector('h1')?.textContent?.trim();
      const author = document.querySelector('[rel="author"], .author, [class*="author"], [itemprop="author"], [data-testid="authorName"]')?.textContent?.trim();
      const date = document.querySelector('time, [datetime], [class*="date"], [class*="Date"], [itemprop="datePublished"]')?.textContent?.trim() ||
                   document.querySelector('time')?.getAttribute('datetime');
      const lead = document.querySelector('.lead, .subtitle, .subheadline, .article-subtitle, [class*="subtitle"], [class*="dek"]')?.textContent?.trim();

      if (title) {
        items.push(this.norm({ name: title, author, time: date, description: lead?.slice(0, 200) }));
      }

      // Extract article sections/subheadings as navigable items
      const article = document.querySelector('article, [role="article"], .article-body, .post-content, .entry-content, #article-body, main');
      if (article) {
        article.querySelectorAll('h2, h3').forEach(h => {
          const name = h.textContent?.trim();
          if (!name || name.length < 3) return;
          let desc = '';
          let el = h.nextElementSibling;
          while (el && !['H2','H3'].includes(el.tagName)) {
            if (el.tagName === 'P' && el.textContent.trim().length > 20) {
              desc = el.textContent.trim().slice(0, 200);
              break;
            }
            el = el.nextElementSibling;
          }
          items.push(this.norm({ name, description: desc || undefined }));
        });
      }
      return items.filter(Boolean);
    },

    cardGeneric(card) {
      if (!card) return null;
      const name = card.querySelector('h1,h2,h3,h4,h5,h6')?.textContent?.trim() ||
                   card.querySelector('a[href]')?.textContent?.trim() ||
                   card.querySelector('strong,b,[class*="title"],[class*="name"]')?.textContent?.trim();
      if (!name || name.length < 2) return null;
      const d = {name};
      const pe = card.querySelector('[class*="price"],[class*="Price"],[data-price]');
      if (pe) d.price = pe.textContent.trim();
      else { const m = card.textContent.match(/[\$\€\£\¥\₹]\s?\d[\d,]*\.?\d*/); if (m) d.price = m[0]; }
      const img = card.querySelector('img[src],img[data-src]');
      if (img) d.image = img.src || img.dataset.src;
      const lnk = card.querySelector('a[href]');
      if (lnk) d.link = lnk.href;
      const rat = card.querySelector('[class*="rating"],[aria-label*="star"]');
      if (rat) d.rating = rat.getAttribute('aria-label') || rat.textContent.trim();
      const desc = card.querySelector('p,[class*="desc"],[class*="snippet"]');
      if (desc) d.description = desc.textContent.trim().slice(0,200);
      const loc = card.querySelector('[class*="location"],address');
      if (loc) d.location = loc.textContent.trim();
      return this.norm(d);
    },

    extractCards() {
      const sels = ['article','[role="listitem"]','.card','.item','.listing','.result','.product',
        '[class*="card"]','[class*="Card"]','[class*="item"]','[class*="Item"]',
        '[class*="product"]','[class*="Product"]','[class*="listing"]','[class*="result"]','[class*="tile"]'];
      let best=null, bc=0;
      for (const s of sels) {
        try {
          const els = Array.from(document.querySelectorAll(s)).filter(e => {
            const r=e.getBoundingClientRect(); return r.width>50&&r.height>50&&e.textContent.trim().length>10;
          });
          if (els.length>=3 && els.length<=200 && els.length>bc && this.similar(els.slice(0,6))) { best=s; bc=els.length; }
        } catch(e){}
      }
      if (!best) return [];
      return Array.from(document.querySelectorAll(best)).map(c=>this.cardGeneric(c)).filter(Boolean);
    },

    similar(els) {
      if (els.length<2) return true;
      const sigs = els.map(e => Array.from(e.children).map(c=>c.tagName).join(','));
      const freq = {};
      sigs.forEach(s => freq[s]=(freq[s]||0)+1);
      return Math.max(...Object.values(freq)) / sigs.length >= 0.5;
    },

    extractRepeated() {
      const containers = document.querySelectorAll('ul,ol,[role="list"],.grid,[class*="grid"],[class*="list"],main,[role="main"]');
      let best=null, bc=0;
      containers.forEach(c => {
        const kids = Array.from(c.children).filter(e => {
          const r=e.getBoundingClientRect(); return r.width>40&&r.height>30&&e.textContent.trim().length>10;
        });
        if (kids.length>=3 && kids.length>bc && this.similar(kids.slice(0,6))) { best=c; bc=kids.length; }
      });
      if (!best) return [];
      return Array.from(best.children).map(e=>this.cardGeneric(e)).filter(Boolean);
    },

    extractTables() {
      const items = [];
      document.querySelectorAll('table').forEach(t => {
        const hdrs = Array.from(t.querySelectorAll('thead th,tr:first-child th')).map(h=>h.textContent.trim());
        if (hdrs.length<2) return;
        t.querySelectorAll('tbody tr').forEach(row => {
          const cells = Array.from(row.querySelectorAll('td'));
          if (cells.length<2) return;
          const d = {};
          cells.forEach((c,i) => d[hdrs[i]||'col'+i]=c.textContent.trim().slice(0,200));
          d.name = d.name || Object.values(d)[0];
          items.push(this.norm(d));
        });
      });
      return items;
    },

    // Universal product link scanner — finds links that look like product pages on any site
    extractProductLinks() {
      const items = [];
      const seen = new Set();
      const currentHost = window.location.hostname;

      document.querySelectorAll('a[href]').forEach(a => {
        const href = a.href;
        if (!href) return;

        // Only internal links
        try { if (new URL(href).hostname !== currentHost) return; } catch(e) { return; }

        // Must look like a product/item page URL
        const path = new URL(href).pathname.toLowerCase();
        if (!path.match(/\/(product|item|goods|p|shop|buy|detail|listing|collection)s?\//i) &&
            !path.match(/\/[a-z0-9-]+-[a-z0-9-]+$/i) // slug-style URLs like /mens-wool-cruiser
           ) return;
        // Skip utility pages
        if (path.match(/\/(cart|login|account|checkout|search|filter|page|tag|categor)/i)) return;
        // Skip if it's the current page
        if (href === window.location.href) return;
        // Skip very short paths (probably not product pages)
        if (path.split('/').filter(Boolean).length < 2) return;

        // Walk up to find card container
        let container = a;
        for (let i = 0; i < 5; i++) {
          if (!container.parentElement) break;
          const parent = container.parentElement;
          if (parent.children.length > 20) break;
          // Stop if container is big enough to be a card (has both text and possibly image)
          const rect = container.getBoundingClientRect();
          if (rect.width > 100 && rect.height > 100) break;
          container = parent;
        }

        // Extract name
        let name = container.querySelector('h1,h2,h3,h4,h5,h6')?.textContent?.trim();
        if (!name) {
          for (const img of container.querySelectorAll('img[alt]')) {
            const alt = img.alt?.trim();
            if (alt && alt.length > 3 && alt.length < 200 && !alt.match(/^(logo|icon|arrow|close|menu|nav)/i)) { name = alt; break; }
          }
        }
        if (!name) {
          // Get the most prominent text that isn't a price
          const texts = [];
          container.querySelectorAll('span,div,p,strong,b').forEach(el => {
            const t = el.textContent?.trim();
            if (t && t.length > 3 && t.length < 200 && el.children.length === 0 &&
                !t.match(/^[\$€£¥₹\d\s,.\-%]+$/) && !t.match(/^(add|buy|shop|sale|new|save|free|sold)/i)) {
              texts.push(t);
            }
          });
          if (texts.length > 0) name = texts.reduce((a, b) => a.length > b.length ? a : b);
        }
        if (!name || name.length < 3) return;

        const nameKey = name.toLowerCase().slice(0, 40);
        if (seen.has(nameKey)) return;
        seen.add(nameKey);

        // Price
        let price = null;
        const priceMatch = container.textContent.match(/\$\s?\d[\d,.]*|€\s?\d[\d,.]*|£\s?\d[\d,.]*|¥\s?\d[\d,.]*/);
        if (priceMatch) price = priceMatch[0];

        // Image
        let image = null;
        const img = container.querySelector('img[src]');
        if (img) image = img.src;

        items.push(this.norm({ name, price, link: href, image }));
      });

      return items.filter(Boolean).slice(0, 60);
    },

    extractMainText() {
      const el = document.querySelector('main,[role="main"],article,[role="article"],#content,.content,.post-content,.entry-content') || document.body;
      const clone = el.cloneNode(true);
      clone.querySelectorAll('script,style,nav,header,footer,aside,[role="navigation"],[class*="cookie"],[class*="modal"],[class*="popup"],[class*="banner"],[class*="sidebar"],[class*="ad-"],[class*="advertisement"],iframe').forEach(x=>x.remove());
      
      // Try to preserve structure: headings + paragraphs
      const parts = [];
      clone.querySelectorAll('h1,h2,h3,h4,p,li,blockquote,pre,td').forEach(el => {
        const text = el.textContent?.trim();
        if (!text || text.length < 5) return;
        const tag = el.tagName;
        if (tag.startsWith('H')) {
          parts.push(`\n## ${text}`);
        } else if (tag === 'BLOCKQUOTE') {
          parts.push(`> ${text.slice(0, 200)}`);
        } else if (tag === 'PRE') {
          parts.push(`[code] ${text.slice(0, 300)}`);
        } else {
          parts.push(text.slice(0, 300));
        }
      });
      
      if (parts.length > 3) {
        return parts.join('\n').slice(0, 4000);
      }
      
      // Fallback: raw text
      return clone.textContent.replace(/\s+/g,' ').trim().slice(0,3000) || null;
    },

    norm(d) {
      if (!d||!d.name) return null;
      for (const k of Object.keys(d)) {
        if (typeof d[k]==='string') { d[k]=d[k].replace(/\s+/g,' ').trim(); if (!d[k]) delete d[k]; }
      }
      if (d.name.length>200) d.name=d.name.slice(0,200)+'...';
      return d;
    },

    dedupe(existing, incoming) {
      const seen = new Set(existing.map(i=>i?.name?.toLowerCase()?.slice(0,50)));
      const merged = [...existing];
      for (const item of incoming) {
        if (item?.name) { const k=item.name.toLowerCase().slice(0,50); if (!seen.has(k)){seen.add(k);merged.push(item);} }
      }
      return merged.filter(Boolean);
    },

    compress(data) {
      const c = {url:data.url, site:data.siteType, title:data.title};
      if (data.meta?.description) c.desc = data.meta.description;
      if (data.navigation?.heading) c.heading = data.navigation.heading;
      if (data.navigation?.breadcrumbs) c.breadcrumbs = data.navigation.breadcrumbs.join(' > ');
      if (data.navigation?.totalResults) c.totalResults = data.navigation.totalResults;
      if (data.items.length > 0) {
        c.items = data.items.slice(0,80).map(i => {
          const s={name:i.name};
          if(i.price)s.price=i.price; if(i.rating)s.rating=i.rating;
          if(i.description)s.desc=i.description.slice(0,80);
          if(i.location)s.location=i.location; if(i.company)s.company=i.company;
          if(i.salary)s.salary=i.salary; if(i.sold)s.sold=i.sold;
          // Only include link in the compressed payload if we have fewer than 40 items
          // (saves tokens — the content script still has all links for click-through)
          if(data.items.length <= 40 && i.link) s.link=i.link;
          return s;
        });
        c.itemCount = data.items.length;
      }
      if (data.structured?.length>0) c.structuredTypes = data.structured.map(s=>s.type);
      if (data.items.length===0 && data.content) c.content = data.content.slice(0,2000);
      return c;
    }
  };

  // =======================================================================
  // WIDGET UI
  // =======================================================================
  function createWidget() {
    const host = document.createElement('div');
    host.id = 'askengine-widget-host';
    host.style.cssText = 'position:fixed;bottom:24px;left:24px;z-index:2147483647;';
    const shadow = host.attachShadow({mode:'open'});
    widgetShadow = shadow;

    shadow.innerHTML = `
      <style>${WIDGET_CSS}</style>
      <link href="https://fonts.googleapis.com/css2?family=DM+Sans:ital,opsz,wght@0,9..40,300;0,9..40,400;0,9..40,500;0,9..40,600;1,9..40,400&family=JetBrains+Mono:wght@400;500&display=swap" rel="stylesheet">
      
      <button class="ae-bubble" id="aeBubble" title="AskEngine · Ctrl+Shift+K">
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.2" stroke-linecap="round" stroke-linejoin="round"><circle cx="11" cy="11" r="7"/><path d="M21 21l-4.35-4.35"/></svg>
        <div class="ae-bubble-ring"></div>
      </button>

      <div class="ae-panel" id="aePanel">
        <div class="ae-header">
          <div class="ae-header-left">
            <div class="ae-wordmark">AskEngine</div>
            <div class="ae-badge" id="aeBadge"></div>
          </div>
          <div class="ae-header-right">
            <div class="ae-status" id="aeStatus">
              <span class="ae-status-dot"></span>
            </div>
            <button class="ae-close" id="aeClose">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round"><path d="M18 6L6 18M6 6l12 12"/></svg>
            </button>
          </div>
        </div>

        <div class="ae-body" id="aeBody">
          <div class="ae-welcome" id="aeWelcome">
            <div class="ae-welcome-icon">
              <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><circle cx="11" cy="11" r="8"/><path d="M21 21l-4.35-4.35"/><path d="M11 8v6M8 11h6" opacity=".4"/></svg>
            </div>
            <div class="ae-welcome-title">What are you looking for?</div>
            <div class="ae-welcome-sub" id="aeSub">Analyzing page content</div>
            <div class="ae-chips" id="aeChips"></div>
          </div>
        </div>

        <div class="ae-footer">
          <div class="ae-input-wrap">
            <svg class="ae-input-icon" width="15" height="15" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round"><circle cx="11" cy="11" r="7"/><path d="M21 21l-4.35-4.35"/></svg>
            <input type="text" class="ae-input" id="aeInput" placeholder="Ask anything about this page..." autocomplete="off" spellcheck="false" />
            <button class="ae-send" id="aeSend" disabled>
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><path d="M5 12h14M12 5l7 7-7 7"/></svg>
            </button>
          </div>
          <div class="ae-footer-meta">
            <span class="ae-powered">Powered by Claude</span>
            <span class="ae-shortcut">⌘⇧K</span>
          </div>
        </div>
      </div>`;

    document.body.appendChild(host);

    shadow.getElementById('aeBubble').addEventListener('click', () => togglePanel());
    shadow.getElementById('aeClose').addEventListener('click', () => togglePanel());
    const input = shadow.getElementById('aeInput');
    const send = shadow.getElementById('aeSend');
    input.addEventListener('input', () => { send.disabled = !input.value.trim(); });
    input.addEventListener('keydown', e => {
      if (e.key === 'Enter' && input.value.trim()) { submitQuery(input.value.trim()); input.value=''; send.disabled=true; }
    });
    send.addEventListener('click', () => {
      if (input.value.trim()) { submitQuery(input.value.trim()); input.value=''; send.disabled=true; }
    });
    document.addEventListener('keydown', e => {
      if ((e.ctrlKey||e.metaKey) && e.shiftKey && (e.key==='K'||e.key==='k')) { e.preventDefault(); togglePanel(); }
    });
  }

  function togglePanel() {
    if (!widgetShadow) return;
    isOpen = !isOpen;
    const panel = widgetShadow.getElementById('aePanel');
    const bubble = widgetShadow.getElementById('aeBubble');
    if (isOpen) {
      panel.classList.add('open');
      bubble.style.transform='scale(0)'; bubble.style.opacity='0';
      widgetShadow.getElementById('aeInput').focus();
      if (!pageData) scanPage();
    } else {
      panel.classList.remove('open');
      bubble.style.transform=''; bubble.style.opacity='';
    }
  }

  async function scanPage() {
    const sub = widgetShadow.getElementById('aeSub');
    const badge = widgetShadow.getElementById('aeBadge');
    sub.textContent = 'Scanning page...';
    try {
      pageData = await Extractor.extractPageContent();
      badge.textContent = pageData.siteType !== 'generic' ? pageData.siteType : window.location.hostname.replace('www.','');
      const n = pageData.items.length;
      sub.textContent = n > 0 ? `Found ${n} items. Ask me anything!` : 'Page loaded. Ask me anything!';
      generateChips();
    } catch(e) { sub.textContent = 'Ready — ask me anything.'; }
  }

  function generateChips() {
    const el = widgetShadow.getElementById('aeChips');
    el.innerHTML = '';
    const s = [];
    if (pageData.items.length > 0) {
      if (pageData.items.some(i=>i.price)) s.push("What's cheapest?");
      if (pageData.items.some(i=>i.rating)) s.push('Highest rated?');
      s.push('Compare top 3');
    } else { s.push('Summarize this page'); s.push('Key points?'); }
    const site = {
      amazon:['Best value?'],temu:['Best deals?','Most popular?'],ebay:['Best deal?'],
      indeed:['Remote jobs?'],reddit:['Top post?','Summarize discussion'],
      twitter:['What are people talking about?','Most engaging post?'],
      youtube:['Summarize this video','Related content?'],
      wikipedia:['Key facts?','Summarize this'],
      hackernews:['Top story?','Trending topics?'],
      stackoverflow:['Best answer?','Summarize solutions'],
      article:['Key takeaways?','Summarize this'],
      medium:['Key takeaways?','Summarize this']
    };
    if (site[pageData.siteType]) s.push(...site[pageData.siteType]);
    s.slice(0,4).forEach(text => {
      const chip = document.createElement('button');
      chip.className = 'ae-chip'; chip.textContent = text;
      chip.addEventListener('click', () => submitQuery(text));
      el.appendChild(chip);
    });
  }

  // =======================================================================
  // QUERY
  // =======================================================================
  async function submitQuery(query) {
    widgetShadow.getElementById('aeWelcome').style.display = 'none';
    addMsg(query, 'user');
    const typing = addMsg('', 'ai', true);

    // Re-scan page before each query to capture lazily loaded content
    try {
      const freshData = await Extractor.extractPageContent();
      if (freshData.items.length > (pageData?.items?.length || 0)) {
        pageData = freshData;
        const badge = widgetShadow.getElementById('aeBadge');
        badge.textContent = pageData.siteType !== 'generic' ? pageData.siteType : window.location.hostname.replace('www.','');
      }
    } catch(e) {}

    const compressed = pageData ? Extractor.compress(pageData) : {url:location.href,title:document.title,content:document.body?.innerText?.slice(0,2000)};
    conversationHistory.push({role:'user', content:query});
    try {
      const resp = await chrome.runtime.sendMessage({
        type:'ASKENGINE_QUERY',
        payload:{query, pageContext:compressed, conversationHistory:conversationHistory.slice(-6), siteType:pageData?.siteType||'generic'}
      });
      typing.remove();

      // Handle free tier limit
      if (resp?.limitReached) {
        showUpgradePrompt();
        return;
      }

      if (resp?.error) addMsg(resp.error, 'ai');
      else if (resp?.answer) {
        conversationHistory.push({role:'assistant',content:resp.answer});
        addMsg(resp.answer,'ai');
        // Update remaining queries indicator
        if (resp.usage && !resp.usage.unlimited) {
          updateQueryCounter(resp.usage.remaining);
        }
      }
      else addMsg('Something went wrong. Please try again.','ai');
    } catch(e) { typing.remove(); addMsg('Connection error. Please try again.','ai'); }
    widgetShadow.getElementById('aeBody').scrollTop = widgetShadow.getElementById('aeBody').scrollHeight;
  }

  function showUpgradePrompt() {
    const body = widgetShadow.getElementById('aeBody');
    const msg = document.createElement('div');
    msg.className = 'ae-msg ae-ai ae-upgrade-prompt';
    msg.innerHTML = `
      <div class="ae-upgrade-inner">
        <div class="ae-upgrade-title">You've used your 10 free queries</div>
        <div class="ae-upgrade-text">Upgrade to AskEngine Pro for unlimited queries on every site.</div>
        <button class="ae-upgrade-btn" id="aeUpgradeStripe">Upgrade — $9.99/month</button>
        <div class="ae-upgrade-divider"><span>or</span></div>
        <button class="ae-upgrade-btn-secondary" id="aeUpgradeBYOK">Use your own API key</button>
        <a class="ae-upgrade-link" href="https://console.anthropic.com/" target="_blank">Get a key from Anthropic →</a>
      </div>
    `;
    body.appendChild(msg);
    body.scrollTop = body.scrollHeight;

    // Stripe upgrade — opens payment link with UID
    msg.querySelector('#aeUpgradeStripe')?.addEventListener('click', async () => {
      const { uid } = await chrome.storage.local.get('uid');
      // STRIPE_PAYMENT_LINK_URL should include ?client_reference_id= parameter
      const paymentUrl = 'https://buy.stripe.com/aFa8wI0O8cKeaG92EO1ZS00' + '?client_reference_id=' + encodeURIComponent(uid);
      window.open(paymentUrl, '_blank');
      
      // Show "checking payment" state
      const btn = msg.querySelector('#aeUpgradeStripe');
      btn.textContent = 'Checking payment status...';
      btn.disabled = true;
      
      // Poll for payment verification
      let attempts = 0;
      const pollInterval = setInterval(async () => {
        attempts++;
        try {
          const resp = await chrome.runtime.sendMessage({ type: 'VERIFY_PAYMENT' });
          if (resp?.paid) {
            clearInterval(pollInterval);
            msg.innerHTML = '<div class="ae-upgrade-inner"><div class="ae-upgrade-title" style="color:var(--ae-accent)">✓ You\'re upgraded!</div><div class="ae-upgrade-text">Unlimited queries unlocked. Ask away.</div></div>';
            const input = widgetShadow.getElementById('aeInput');
            const send = widgetShadow.getElementById('aeSend');
            if (input) { input.disabled = false; input.placeholder = 'Ask anything about this page...'; }
            if (send) send.disabled = false;
          }
        } catch(e) {}
        if (attempts > 60) clearInterval(pollInterval); // Stop after 5 min
      }, 5000);
    });

    // BYOK — open settings
    msg.querySelector('#aeUpgradeBYOK')?.addEventListener('click', () => {
      chrome.runtime.openOptionsPage?.() || window.open(chrome.runtime.getURL('options.html'));
    });

    // Disable input
    const input = widgetShadow.getElementById('aeInput');
    const send = widgetShadow.getElementById('aeSend');
    if (input) { input.disabled = true; input.placeholder = 'Upgrade to continue asking...'; }
    if (send) send.disabled = true;
  }

  function updateQueryCounter(remaining) {
    let counter = widgetShadow.getElementById('aeQueryCounter');
    if (!counter) {
      counter = document.createElement('span');
      counter.id = 'aeQueryCounter';
      counter.className = 'ae-query-counter';
      const meta = widgetShadow.querySelector('.ae-footer-meta');
      if (meta) {
        const powered = meta.querySelector('.ae-powered');
        if (powered) powered.after(counter);
      }
    }
    if (remaining <= 5) {
      counter.textContent = `${remaining} free left`;
      counter.style.color = remaining <= 2 ? 'rgba(255,120,80,.7)' : 'rgba(255,255,255,.25)';
    } else {
      counter.textContent = '';
    }
  }

  function addMsg(text, role, isTyping=false) {
    const body = widgetShadow.getElementById('aeBody');
    const msg = document.createElement('div');
    msg.className = `ae-msg ae-${role}`;
    if (isTyping) {
      msg.innerHTML = '<div class="ae-dots"><span></span><span></span><span></span></div>';
    } else if (role === 'ai' && pageData?.items?.length > 0) {
      // Build a lookup of item names → link URLs
      const itemMap = [];
      for (const item of pageData.items) {
        if (item.name && item.link) {
          itemMap.push({ name: item.name, link: item.link, price: item.price, image: item.image });
        }
      }

      // First do basic markdown formatting
      let html = text
        .replace(/`(.*?)`/g, '<code>$1</code>')
        .replace(/\n/g, '<br>');

      // Replace **bold text** — check if each bold phrase matches an item name
      html = html.replace(/\*\*(.*?)\*\*/g, (match, boldText) => {
        const found = findMatchingItem(boldText, itemMap);
        if (found) {
          return `<a class="ae-item-link" href="${escapeAttr(found.link)}" target="_blank" title="${escapeAttr(found.price || '')}">${escapeHtml(boldText)}<span class="ae-link-arrow">↗</span></a>`;
        }
        return `<strong>${escapeHtml(boldText)}</strong>`;
      });

      // Also replace *italic*
      html = html.replace(/\*(.*?)\*/g, '<em>$1</em>');

      msg.innerHTML = html;

      // Attach click handlers (stopPropagation so shadow DOM clicks work)
      msg.querySelectorAll('.ae-item-link').forEach(link => {
        link.addEventListener('click', (e) => {
          e.preventDefault();
          window.open(link.getAttribute('href'), '_blank');
        });
      });
    } else {
      msg.innerHTML = text
        .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
        .replace(/\*(.*?)\*/g, '<em>$1</em>')
        .replace(/`(.*?)`/g, '<code>$1</code>')
        .replace(/\n/g, '<br>');
    }
    body.appendChild(msg);
    body.scrollTop = body.scrollHeight;
    return msg;
  }

  function findMatchingItem(boldText, itemMap) {
    const query = boldText.toLowerCase().trim();
    
    // Skip obvious non-product bold text
    if (query.length < 3) return null;
    if (query.match(/^\$[\d.,]+$|^\d+%|^best |^top |^great |^note|^tip|^warning/i)) return null;
    
    // Pass 1: Exact match
    for (const item of itemMap) {
      if (item.name.toLowerCase() === query) return item;
    }
    
    // Pass 2: Containment (bold text inside item name OR item name inside bold text)
    for (const item of itemMap) {
      const itemLower = item.name.toLowerCase();
      if (itemLower.includes(query) || query.includes(itemLower)) return item;
    }

    // Pass 3: Key-phrase match — extract significant words (3+ chars, not common)
    const stopWords = new Set(['the','and','for','with','that','this','from','have','are','was','were','been','being','its','but','not','you','all','can','had','her','one','our','out','has','than','too','any','each','per','also','very','just','more','most','some','such','only','into','over','other','new','set','pack','pair']);
    const queryWords = query.split(/[\s\-–,&+/|]+/).filter(w => w.length >= 3 && !stopWords.has(w));
    
    if (queryWords.length >= 1) {
      let bestMatch = null, bestScore = 0;
      
      for (const item of itemMap) {
        const itemLower = item.name.toLowerCase();
        const itemWords = itemLower.split(/[\s\-–,&+/|]+/).filter(w => w.length >= 3 && !stopWords.has(w));
        
        // Count how many query words appear in the item name (substring match, not just whole word)
        const matched = queryWords.filter(qw => itemLower.includes(qw)).length;
        
        // Also count reverse — how many item words appear in the query
        const reverseMatched = itemWords.filter(iw => query.includes(iw)).length;
        
        // Score: prioritize forward match but consider reverse too
        const forwardScore = queryWords.length > 0 ? matched / queryWords.length : 0;
        const reverseScore = itemWords.length > 0 ? reverseMatched / itemWords.length : 0;
        const combinedScore = (forwardScore * 0.7) + (reverseScore * 0.3);
        
        // Need at least 2 matching words or 40% combined score
        if ((matched >= 2 || forwardScore >= 0.4) && combinedScore > bestScore) {
          bestScore = combinedScore;
          bestMatch = item;
        }
      }
      if (bestMatch && bestScore >= 0.3) return bestMatch;
    }

    // Pass 4: Number-aware matching — if bold text contains specific numbers/measurements
    // that appear in only one item, that's likely a match (e.g. "120 Pack" → "120 Pack Reusable Cable Ties")
    const numbers = query.match(/\d+/g);
    if (numbers && numbers.length > 0) {
      const candidates = itemMap.filter(item => {
        const itemLower = item.name.toLowerCase();
        return numbers.every(n => itemLower.includes(n));
      });
      // If numbers narrow it to 1-2 items, pick the best text overlap
      if (candidates.length === 1) return candidates[0];
      if (candidates.length > 1 && candidates.length <= 3) {
        let best = null, bestOverlap = 0;
        for (const c of candidates) {
          const overlap = queryWords?.filter(w => c.name.toLowerCase().includes(w))?.length || 0;
          if (overlap > bestOverlap) { bestOverlap = overlap; best = c; }
        }
        if (best) return best;
      }
    }
    
    return null;
  }

  function escapeHtml(str) {
    return str.replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;');
  }

  function escapeAttr(str) {
    return str.replace(/&/g,'&amp;').replace(/"/g,'&quot;').replace(/'/g,'&#39;').replace(/</g,'&lt;').replace(/>/g,'&gt;');
  }

  // =======================================================================
  // CSS
  // =======================================================================
  const WIDGET_CSS = `
    @import url('https://fonts.googleapis.com/css2?family=DM+Sans:ital,opsz,wght@0,9..40,300;0,9..40,400;0,9..40,500;0,9..40,600;1,9..40,400&family=JetBrains+Mono:wght@400;500&display=swap');

    *{box-sizing:border-box;margin:0;padding:0}

    /* ── Floating Trigger ── */
    .ae-bubble{
      position:relative;width:52px;height:52px;border-radius:16px;
      background:linear-gradient(135deg,#1a1a24 0%,#22222e 100%);
      border:1px solid rgba(255,255,255,.13);
      cursor:pointer;display:flex;align-items:center;justify-content:center;
      color:#fff;
      box-shadow:
        0 0 0 1px rgba(0,0,0,.5),
        0 2px 8px rgba(0,0,0,.4),
        0 12px 40px rgba(0,0,0,.3),
        0 0 20px rgba(0,229,160,.12),
        0 0 60px rgba(0,229,160,.06),
        inset 0 1px 0 rgba(255,255,255,.1);
      transition:all .4s cubic-bezier(.4,0,.2,1);
      animation:aeGlow 4s ease-in-out infinite;
    }
    @keyframes aeGlow{
      0%,100%{box-shadow:0 0 0 1px rgba(0,0,0,.5),0 2px 8px rgba(0,0,0,.4),0 12px 40px rgba(0,0,0,.3),0 0 20px rgba(0,229,160,.12),0 0 60px rgba(0,229,160,.06),inset 0 1px 0 rgba(255,255,255,.1)}
      50%{box-shadow:0 0 0 1px rgba(0,0,0,.5),0 2px 8px rgba(0,0,0,.4),0 12px 40px rgba(0,0,0,.3),0 0 28px rgba(0,229,160,.22),0 0 80px rgba(0,229,160,.1),inset 0 1px 0 rgba(255,255,255,.1)}
    }
    .ae-bubble:hover{
      transform:translateY(-2px);
      animation:none;
      box-shadow:
        0 0 0 1px rgba(0,0,0,.5),
        0 4px 12px rgba(0,0,0,.4),
        0 20px 50px rgba(0,0,0,.35),
        0 0 30px rgba(0,229,160,.28),
        0 0 80px rgba(0,229,160,.14),
        inset 0 1px 0 rgba(255,255,255,.12);
      border-color:rgba(0,229,160,.25);
    }
    .ae-bubble:active{transform:translateY(0) scale(.96)}
    .ae-bubble svg{position:relative;z-index:1;opacity:.9}
    .ae-bubble-ring{
      position:absolute;inset:-3px;border-radius:19px;
      border:1.5px solid transparent;
      background:linear-gradient(135deg,rgba(0,229,160,.3),rgba(0,229,160,0),rgba(0,229,160,.15)) border-box;
      -webkit-mask:linear-gradient(#fff 0 0) padding-box,linear-gradient(#fff 0 0);
      -webkit-mask-composite:xor;mask-composite:exclude;
      opacity:0;transition:opacity .4s;
    }
    .ae-bubble:hover .ae-bubble-ring{opacity:1}

    /* ── Panel ── */
    .ae-panel{
      position:fixed;bottom:24px;right:24px;
      width:400px;max-height:580px;
      background:rgba(12,12,18,.97);
      backdrop-filter:blur(40px) saturate(1.4);
      -webkit-backdrop-filter:blur(40px) saturate(1.4);
      border:1px solid rgba(255,255,255,.07);
      border-radius:20px;
      display:flex;flex-direction:column;
      box-shadow:
        0 0 0 1px rgba(0,0,0,.3),
        0 8px 24px rgba(0,0,0,.35),
        0 32px 80px rgba(0,0,0,.4);
      transform:translateY(12px) scale(0.97);
      opacity:0;pointer-events:none;
      transition:all .35s cubic-bezier(.16,1,.3,1);
      font-family:'DM Sans',-apple-system,BlinkMacSystemFont,system-ui,sans-serif;
      font-size:13.5px;color:#e4e4ec;
      overflow:hidden;
    }
    .ae-panel.open{
      transform:translateY(0) scale(1);opacity:1;pointer-events:all;
    }

    /* ── Header ── */
    .ae-header{
      padding:16px 20px;display:flex;align-items:center;justify-content:space-between;
      border-bottom:1px solid rgba(255,255,255,.05);
      background:rgba(255,255,255,.02);
    }
    .ae-header-left{display:flex;align-items:center;gap:10px}
    .ae-header-right{display:flex;align-items:center;gap:6px}
    .ae-wordmark{
      font-family:'DM Sans',sans-serif;
      font-weight:600;font-size:14.5px;letter-spacing:-.02em;
      color:#fff;
    }
    .ae-badge{
      font-family:'JetBrains Mono',monospace;
      font-size:10px;font-weight:500;letter-spacing:.03em;
      padding:3px 8px;border-radius:6px;
      background:rgba(0,229,160,.1);
      color:rgba(0,229,160,.9);
      text-transform:uppercase;
      line-height:1;
    }
    .ae-status{display:flex;align-items:center;gap:5px;padding:0 6px}
    .ae-status-dot{
      width:6px;height:6px;border-radius:50%;
      background:#00e5a0;
      box-shadow:0 0 6px rgba(0,229,160,.5);
      animation:aePulse 3s ease-in-out infinite;
    }
    @keyframes aePulse{0%,100%{opacity:1}50%{opacity:.4}}
    .ae-close{
      background:none;border:none;color:rgba(255,255,255,.3);cursor:pointer;
      width:30px;height:30px;border-radius:8px;
      display:flex;align-items:center;justify-content:center;
      transition:all .2s;
    }
    .ae-close:hover{background:rgba(255,255,255,.06);color:rgba(255,255,255,.7)}

    /* ── Body ── */
    .ae-body{
      flex:1;overflow-y:auto;padding:20px;
      min-height:200px;max-height:380px;
    }
    .ae-body::-webkit-scrollbar{width:3px}
    .ae-body::-webkit-scrollbar-track{background:transparent}
    .ae-body::-webkit-scrollbar-thumb{background:rgba(255,255,255,.08);border-radius:3px}
    .ae-body::-webkit-scrollbar-thumb:hover{background:rgba(255,255,255,.14)}

    /* ── Welcome ── */
    .ae-welcome{
      display:flex;flex-direction:column;align-items:center;text-align:center;
      padding:24px 12px 8px;gap:10px;
    }
    .ae-welcome-icon{
      color:rgba(255,255,255,.2);
      margin-bottom:4px;
    }
    .ae-welcome-title{
      font-size:17px;font-weight:500;letter-spacing:-.02em;
      color:rgba(255,255,255,.95);
    }
    .ae-welcome-sub{
      font-size:12.5px;color:rgba(255,255,255,.35);
      font-weight:400;
    }
    .ae-chips{
      display:flex;flex-wrap:wrap;gap:6px;justify-content:center;
      margin-top:16px;
    }
    .ae-chip{
      font-family:'DM Sans',sans-serif;
      font-size:12px;font-weight:500;letter-spacing:-.01em;
      padding:7px 16px;border-radius:100px;
      border:1px solid rgba(255,255,255,.08);
      background:rgba(255,255,255,.03);
      color:rgba(255,255,255,.6);
      cursor:pointer;transition:all .25s cubic-bezier(.4,0,.2,1);
    }
    .ae-chip:hover{
      border-color:rgba(0,229,160,.35);
      background:rgba(0,229,160,.06);
      color:rgba(0,229,160,.95);
      transform:translateY(-1px);
    }
    .ae-chip:active{transform:translateY(0) scale(.97)}

    /* ── Messages ── */
    .ae-msg{
      margin-bottom:12px;padding:12px 16px;border-radius:14px;
      line-height:1.6;max-width:90%;
      animation:aeSlideIn .3s cubic-bezier(.16,1,.3,1);
      word-wrap:break-word;font-weight:400;
    }
    @keyframes aeSlideIn{
      from{opacity:0;transform:translateY(6px)}
      to{opacity:1;transform:translateY(0)}
    }
    .ae-user{
      background:linear-gradient(135deg,#00e5a0,#00cc8e);
      color:#071a12;margin-left:auto;
      font-weight:500;letter-spacing:-.01em;
      border-bottom-right-radius:6px;
      box-shadow:0 2px 12px rgba(0,229,160,.2);
    }
    .ae-ai{
      background:rgba(255,255,255,.04);
      border:1px solid rgba(255,255,255,.06);
      border-bottom-left-radius:6px;
      color:rgba(255,255,255,.85);
    }
    .ae-ai strong{color:#00e5a0;font-weight:600}
    .ae-ai code{
      font-family:'JetBrains Mono',monospace;
      background:rgba(255,255,255,.06);
      padding:2px 6px;border-radius:4px;font-size:12px;
    }

    /* ── Clickable Item Links ── */
    .ae-item-link{
      color:#00e5a0;text-decoration:none;font-weight:600;
      cursor:pointer;
      background:linear-gradient(to bottom,transparent 60%,rgba(0,229,160,.1) 60%);
      padding-bottom:0px;
      transition:all .2s cubic-bezier(.4,0,.2,1);
      display:inline;
      border-radius:2px;
    }
    .ae-item-link:hover{
      background:linear-gradient(to bottom,transparent 0%,rgba(0,229,160,.12) 0%);
      text-shadow:0 0 20px rgba(0,229,160,.2);
    }
    .ae-link-arrow{
      font-size:10px;margin-left:3px;opacity:.5;
      display:inline-block;transition:all .2s cubic-bezier(.4,0,.2,1);
      font-family:'JetBrains Mono',monospace;
    }
    .ae-item-link:hover .ae-link-arrow{
      transform:translate(2px,-2px);opacity:.9;
    }

    /* ── Typing Dots ── */
    .ae-dots{display:flex;gap:5px;padding:6px 2px;align-items:center}
    .ae-dots span{
      width:5px;height:5px;border-radius:50%;
      background:rgba(255,255,255,.25);
      animation:aeDot 1.2s ease-in-out infinite;
    }
    .ae-dots span:nth-child(2){animation-delay:.15s}
    .ae-dots span:nth-child(3){animation-delay:.3s}
    @keyframes aeDot{
      0%,80%,100%{transform:scale(1);opacity:.3}
      40%{transform:scale(1.3);opacity:.8}
    }

    /* ── Footer ── */
    .ae-footer{
      padding:14px 16px 12px;
      border-top:1px solid rgba(255,255,255,.05);
      background:rgba(255,255,255,.015);
    }
    .ae-input-wrap{
      display:flex;align-items:center;gap:0;
      background:rgba(255,255,255,.05);
      border:1px solid rgba(255,255,255,.07);
      border-radius:14px;
      padding:0 4px 0 14px;
      transition:all .25s cubic-bezier(.4,0,.2,1);
    }
    .ae-input-wrap:focus-within{
      border-color:rgba(0,229,160,.3);
      background:rgba(255,255,255,.06);
      box-shadow:0 0 0 3px rgba(0,229,160,.06);
    }
    .ae-input-icon{
      color:rgba(255,255,255,.2);flex-shrink:0;
      transition:color .2s;
    }
    .ae-input-wrap:focus-within .ae-input-icon{color:rgba(0,229,160,.5)}
    .ae-input{
      flex:1;padding:11px 10px;
      border:none;background:transparent;
      color:#fff;font-size:13.5px;
      font-family:'DM Sans',sans-serif;
      font-weight:400;letter-spacing:-.01em;
      outline:none;
    }
    .ae-input::placeholder{color:rgba(255,255,255,.25);font-weight:400}
    .ae-send{
      width:34px;height:34px;border-radius:10px;border:none;
      background:rgba(0,229,160,.15);
      color:rgba(0,229,160,.6);
      cursor:pointer;display:flex;align-items:center;justify-content:center;
      transition:all .2s cubic-bezier(.4,0,.2,1);
      flex-shrink:0;
    }
    .ae-send:disabled{opacity:.2;cursor:not-allowed}
    .ae-send:not(:disabled){
      background:#00e5a0;color:#071a12;
    }
    .ae-send:hover:not(:disabled){
      transform:scale(1.05);
      box-shadow:0 2px 12px rgba(0,229,160,.3);
    }
    .ae-send:active:not(:disabled){transform:scale(.95)}

    .ae-footer-meta{
      display:flex;align-items:center;justify-content:space-between;
      padding:8px 4px 0;
    }
    .ae-powered{
      font-size:10.5px;color:rgba(255,255,255,.15);
      font-weight:400;letter-spacing:.01em;
    }
    .ae-shortcut{
      font-family:'JetBrains Mono',monospace;
      font-size:10px;color:rgba(255,255,255,.12);
      background:rgba(255,255,255,.04);
      padding:2px 6px;border-radius:4px;
      border:1px solid rgba(255,255,255,.04);
    }

    /* ── Upgrade Prompt ── */
    .ae-upgrade-prompt{
      background:rgba(255,255,255,.03) !important;
      border:1px solid rgba(0,229,160,.15) !important;
      max-width:100% !important;
    }
    .ae-upgrade-inner{display:flex;flex-direction:column;gap:10px;text-align:center;padding:8px 4px}
    .ae-upgrade-title{font-size:14px;font-weight:600;color:#fff;letter-spacing:-.01em}
    .ae-upgrade-text{font-size:12.5px;color:rgba(255,255,255,.45);line-height:1.55}
    .ae-upgrade-btn{
      margin-top:4px;padding:10px 20px;border-radius:10px;border:none;
      background:linear-gradient(135deg,#00e5a0,#00cc8e);
      color:#071a12;font-size:13px;font-weight:600;
      font-family:'DM Sans',sans-serif;letter-spacing:-.01em;
      cursor:pointer;transition:all .2s cubic-bezier(.4,0,.2,1);
    }
    .ae-upgrade-btn:hover{transform:translateY(-1px);box-shadow:0 4px 16px rgba(0,229,160,.25)}
    .ae-upgrade-divider{
      display:flex;align-items:center;gap:12px;
      font-size:11px;color:rgba(255,255,255,.2);
    }
    .ae-upgrade-divider::before,.ae-upgrade-divider::after{
      content:'';flex:1;height:1px;background:rgba(255,255,255,.06);
    }
    .ae-upgrade-btn-secondary{
      padding:9px 18px;border-radius:10px;
      border:1px solid rgba(255,255,255,.08);background:rgba(255,255,255,.03);
      color:rgba(255,255,255,.5);font-size:12.5px;font-weight:500;
      font-family:'DM Sans',sans-serif;cursor:pointer;
      transition:all .2s;width:100%;
    }
    .ae-upgrade-btn-secondary:hover{border-color:rgba(255,255,255,.15);color:rgba(255,255,255,.7)}
    .ae-upgrade-link{
      font-size:11.5px;color:rgba(0,229,160,.6);
      text-decoration:none;transition:color .2s;
    }
    .ae-upgrade-link:hover{color:rgba(0,229,160,.9)}

    /* ── Query Counter ── */
    .ae-query-counter{
      font-family:'JetBrains Mono',monospace;
      font-size:10px;color:rgba(255,255,255,.25);
      margin-left:8px;
    }

    /* ── Responsive ── */
    @media(max-width:480px){
      .ae-panel{width:calc(100vw - 16px);right:8px;bottom:8px;border-radius:16px}
    }
  `;

  // =======================================================================
  // INIT
  // =======================================================================
  function init() {
    const p = window.location.protocol;
    if (p==='chrome:'||p==='chrome-extension:'||p==='about:'||p==='moz-extension:') return;
    createWidget();
  }

  if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', init);
  else init();

  // Listen for popup messages
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    if (msg.type === 'OPEN_WIDGET' || msg.type === 'TOGGLE_WIDGET') {
      if (!isOpen) togglePanel();
      sendResponse({ok:true});
    }
    if (msg.type === 'TOGGLE_ENABLED') {
      const host = document.getElementById('askengine-widget-host');
      if (host) host.style.display = msg.enabled===false ? 'none' : '';
      sendResponse({ok:true});
    }
    return false;
  });
})();
